/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.client.render.terrain;

import com.mojang.logging.CategorizedLogger;
import com.mojang.logging.LogUtils;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import net.minecraft.client.render.terrain.VertexBuffer;
import net.minecraft.client.render.terrain.VertexConfig;
import net.minecraft.client.render.terrain.VertexData;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.BufferUtils;
import org.lwjgl.opengl.GL41;

public class RenderList<V extends VertexConfig<V>> {
    private static final CategorizedLogger LOGGER = LogUtils.getLogger();
    @NotNull
    private final V vertexConfig;
    @NotNull
    private final VertexBuffer vertexBuffer;
    @NotNull
    private final @NotNull List<@NotNull VertexBuffer.Entry<V>> entryList = new ArrayList<VertexBuffer.Entry<V>>();
    @NotNull
    private final @NotNull Set<@NotNull VertexBuffer.Entry<V>> visibleEntries = new HashSet<VertexBuffer.Entry<V>>();
    @NotNull
    private IntBuffer sizeBuffer;
    @NotNull
    private IntBuffer posBuffer;
    private boolean visibleEntriesUpdated = true;

    public RenderList(@NotNull VertexBuffer vertexBuffer, @NotNull V vertexConfig, int capacity) {
        this.vertexBuffer = vertexBuffer;
        this.vertexConfig = vertexConfig;
        this.posBuffer = BufferUtils.createByteBuffer(capacity * 4).asIntBuffer();
        this.sizeBuffer = BufferUtils.createByteBuffer(capacity * 4).asIntBuffer();
    }

    public void draw() {
        if (this.entryList.isEmpty()) {
            return;
        }
        if (this.visibleEntriesUpdated) {
            int vertexSize = this.vertexConfig.getVertexSize();
            this.posBuffer = RenderList.expandIfNecessary(this.posBuffer, this.visibleEntries.size());
            this.sizeBuffer = RenderList.expandIfNecessary(this.sizeBuffer, this.visibleEntries.size());
            this.posBuffer.clear();
            this.sizeBuffer.clear();
            for (VertexBuffer.Entry<V> entry : this.visibleEntries) {
                int pos = (int)(entry.position / (long)vertexSize);
                int size = (int)(entry.size / (long)vertexSize);
                assert (pos >= 0) : "Entry pos must be positive!";
                assert (size >= 0) : "Entry size must be positive!";
                this.posBuffer.put(pos);
                this.sizeBuffer.put(size);
            }
            this.posBuffer.flip();
            this.sizeBuffer.flip();
            this.visibleEntriesUpdated = false;
        }
        if (this.visibleEntries.isEmpty()) {
            return;
        }
        GL41.glMultiDrawArrays(this.vertexConfig.drawMode().cap, this.posBuffer, this.sizeBuffer);
    }

    @Nullable
    public VertexBuffer.Entry<V> add(@NotNull VertexData<V> vertexData, boolean visible) {
        if (vertexData.vertexCount <= 0) {
            return null;
        }
        @NotNull VertexBuffer.Entry<V> entry = Objects.requireNonNull(this.vertexBuffer.addVertexData(vertexData));
        this.entryList.add(entry);
        if (visible) {
            this.visibleEntries.add(entry);
            this.visibleEntriesUpdated = true;
        }
        return entry;
    }

    public void setVisible(@NotNull VertexBuffer.Entry<V> entry, boolean visible) {
        if (visible) {
            if (this.visibleEntries.add(entry)) {
                this.visibleEntriesUpdated = true;
            }
        } else if (this.visibleEntries.remove(entry)) {
            this.visibleEntriesUpdated = true;
        }
    }

    public boolean remove(@NotNull VertexBuffer.Entry<V> entry) {
        if (this.entryList.remove(entry)) {
            this.vertexBuffer.removeEntry(entry);
            this.visibleEntries.remove(entry);
            this.visibleEntriesUpdated = true;
            return true;
        }
        return false;
    }

    public int size() {
        return this.entryList.size();
    }

    public int visibleEntryCount() {
        return this.visibleEntries.size();
    }

    @NotNull
    public V getVertexConfig() {
        return this.vertexConfig;
    }

    @NotNull
    private static IntBuffer expandIfNecessary(@NotNull IntBuffer buffer, int requiredCapacity) {
        if (buffer.capacity() < requiredCapacity) {
            int newCapacity = Math.max(buffer.capacity() * 2, requiredCapacity);
            LOGGER.warn("Expanding RenderList IntBuffer from {} to {} ints...", (Object)buffer.capacity(), (Object)newCapacity);
            return BufferUtils.createIntBuffer(newCapacity);
        }
        return buffer;
    }
}

