/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.core.world.save;

import com.mojang.logging.LogUtils;
import com.mojang.nbt.NbtIo;
import com.mojang.nbt.tags.CompoundTag;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.List;
import java.util.UUID;
import net.minecraft.core.MinecraftException;
import net.minecraft.core.entity.player.Player;
import net.minecraft.core.world.Dimension;
import net.minecraft.core.world.chunk.ChunkLoader;
import net.minecraft.core.world.chunk.ChunkLoaderLegacy;
import net.minecraft.core.world.save.DimensionData;
import net.minecraft.core.world.save.ISaveFormat;
import net.minecraft.core.world.save.LevelData;
import net.minecraft.core.world.save.LevelStorage;
import net.minecraft.core.world.save.PlayerIO;
import net.minecraft.core.world.saveddata.SavedDataStorage;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

public abstract class LevelStorageBase
implements LevelStorage {
    private static final Logger LOGGER = LogUtils.getLogger();
    protected final String worldDirName;
    protected final File saveDirectory;
    protected final File playersDirectory;
    protected final File dataDirectory;
    protected final long now = System.currentTimeMillis();
    protected final ISaveFormat saveFormat;
    @NotNull
    private final SavedDataStorage savedDataStorage;

    public LevelStorageBase(ISaveFormat saveFormat, File savesDir, String worldDirName) {
        this.worldDirName = worldDirName;
        this.saveFormat = saveFormat;
        this.saveDirectory = new File(savesDir, worldDirName);
        this.saveDirectory.mkdirs();
        this.playersDirectory = new File(this.saveDirectory, "players");
        this.dataDirectory = new File(this.saveDirectory, "data");
        this.dataDirectory.mkdirs();
        this.playersDirectory.mkdirs();
        this.lockSession();
        this.savedDataStorage = new SavedDataStorage(this);
    }

    private void lockSession() {
        try {
            File file = new File(this.saveDirectory, "session.lock");
            try (DataOutputStream stream = new DataOutputStream(Files.newOutputStream(file.toPath(), new OpenOption[0]));){
                stream.writeLong(this.now);
            }
        }
        catch (IOException ioexception) {
            LOGGER.error("Failed to check session lock!", ioexception);
            throw new RuntimeException("Failed to check session lock, aborting");
        }
    }

    protected File getSaveDirectory() {
        return this.saveDirectory;
    }

    @Override
    public void checkSessionLock() {
        try {
            File file = new File(this.saveDirectory, "session.lock");
            try (DataInputStream datainputstream = new DataInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]));){
                if (datainputstream.readLong() != this.now) {
                    throw new MinecraftException("The save is being accessed from another location, aborting");
                }
            }
        }
        catch (IOException ioexception) {
            throw new MinecraftException("Failed to check session lock, aborting");
        }
    }

    @Override
    @Nullable
    public ChunkLoader getChunkLoader(@NotNull Dimension dimension) {
        File dimDir = this.saveFormat.getDimensionRootDir(this.worldDirName, dimension);
        dimDir.mkdirs();
        return new ChunkLoaderLegacy(dimDir, true);
    }

    @Override
    @Nullable
    public LevelData getLevelData() {
        return this.saveFormat.getLevelData(this.worldDirName);
    }

    @Override
    @Nullable
    public CompoundTag getLevelDataRaw() {
        return this.saveFormat.getLevelDataRaw(this.worldDirName);
    }

    @Override
    @Nullable
    public DimensionData getDimensionData(@NotNull Dimension dimension) {
        return this.saveFormat.getDimensionData(this.worldDirName, dimension);
    }

    @Override
    @Nullable
    public CompoundTag getDimensionDataRaw(@NotNull Dimension dimension) {
        return this.saveFormat.getDimensionDataRaw(this.worldDirName, dimension);
    }

    @Override
    public void saveLevelDataAndPlayerData(@NotNull LevelData levelData, @NotNull List<Player> playerList) {
        CompoundTag dataTag = LevelData.toCompoundTag(levelData, new CompoundTag());
        CompoundTag savedTag = new CompoundTag();
        savedTag.put("Data", dataTag);
        try {
            File file = new File(this.saveDirectory, "level.dat_new");
            File file1 = new File(this.saveDirectory, "level.dat_old");
            File file2 = new File(this.saveDirectory, "level.dat");
            NbtIo.writeCompressed(savedTag, Files.newOutputStream(file.toPath(), new OpenOption[0]));
            if (file1.exists()) {
                file1.delete();
            }
            file2.renameTo(file1);
            if (file2.exists()) {
                file2.delete();
            }
            file.renameTo(file2);
            if (file.exists()) {
                file.delete();
            }
        }
        catch (Exception exception) {
            LOGGER.error("Failed to save level data to disk!", exception);
        }
        for (Player player : playerList) {
            try {
                this.save(player);
            }
            catch (Exception e) {
                LOGGER.error("Failed to save player data for player '{}'!", (Object)player.username, (Object)e);
            }
        }
        this.savedDataStorage.save();
    }

    @Override
    public void saveLevelData(@NotNull LevelData levelData) {
        this.saveLevelDataRaw(LevelData.toCompoundTag(levelData, new CompoundTag()));
    }

    @Override
    public void saveLevelDataRaw(@NotNull CompoundTag levelDataTag) {
        CompoundTag rootTag = new CompoundTag();
        rootTag.put("Data", levelDataTag);
        try {
            File levelDatNew = new File(this.saveDirectory, "level.dat_new");
            File levelDatOld = new File(this.saveDirectory, "level.dat_old");
            File levelDat = new File(this.saveDirectory, "level.dat");
            NbtIo.writeCompressed(rootTag, Files.newOutputStream(levelDatNew.toPath(), new OpenOption[0]));
            if (levelDatOld.exists()) {
                levelDatOld.delete();
            }
            levelDat.renameTo(levelDatOld);
            if (levelDat.exists()) {
                levelDat.delete();
            }
            levelDatNew.renameTo(levelDat);
            if (levelDatNew.exists()) {
                levelDatNew.delete();
            }
        }
        catch (Exception exception) {
            LOGGER.error("Failed to save level data to disk!", exception);
        }
    }

    @Override
    public void saveDimensionData(int dimensionId, @NotNull DimensionData dimensionData) {
        this.saveDimensionDataRaw(dimensionId, DimensionData.toCompoundTag(dimensionData, new CompoundTag()));
    }

    @Override
    public void saveDimensionDataRaw(int dimensionId, @NotNull CompoundTag dimensionDataTag) {
        CompoundTag rootTag = new CompoundTag();
        rootTag.put("Data", dimensionDataTag);
        File dimensionsDirectory = new File(this.saveDirectory, "dimensions");
        File dimensionDirectory = new File(dimensionsDirectory, Integer.toString(dimensionId));
        try {
            File dimensionDatNew = new File(dimensionDirectory, "dimension.dat_new");
            File dimensionDatOld = new File(dimensionDirectory, "dimension.dat_old");
            File dimensionDat = new File(dimensionDirectory, "dimension.dat");
            if (!dimensionDirectory.exists()) {
                dimensionDirectory.mkdirs();
            }
            NbtIo.writeCompressed(rootTag, Files.newOutputStream(dimensionDatNew.toPath(), new OpenOption[0]));
            if (dimensionDatOld.exists()) {
                dimensionDatOld.delete();
            }
            dimensionDat.renameTo(dimensionDatOld);
            if (dimensionDat.exists()) {
                dimensionDat.delete();
            }
            dimensionDatNew.renameTo(dimensionDat);
            if (dimensionDatNew.exists()) {
                dimensionDatNew.delete();
            }
        }
        catch (Exception exception) {
            LOGGER.error("Failed to save dimension {} data to disk!", (Object)dimensionId, (Object)exception);
        }
    }

    @Override
    public File getDataFile(@NotNull String fileName) {
        return new File(this.dataDirectory, fileName + ".dat");
    }

    @Override
    public void save(@NotNull Player player) {
        try {
            CompoundTag compoundTag = new CompoundTag();
            player.saveWithoutId(compoundTag);
            compoundTag.putInt("Format", 1);
            File temp = new File(this.playersDirectory, "_tmp_.dat");
            File save = new File(this.playersDirectory, String.valueOf(player.uuid) + ".dat");
            NbtIo.writeCompressed(compoundTag, new FileOutputStream(temp));
            if (save.exists()) {
                save.delete();
            }
            temp.renameTo(save);
        }
        catch (Exception exception) {
            LOGGER.warn("Failed to save player data for " + player.username);
        }
    }

    @Override
    public void load(@NotNull Player player) {
        CompoundTag tag = this.getPlayerData(player.username, player.uuid);
        if (tag != null) {
            player.load(tag);
            return;
        }
        LevelData data = this.getLevelData();
        if (data != null && (tag = data.getPlayerTag()) != null) {
            player.load(tag);
        }
    }

    @Override
    public PlayerIO getPlayerFileData() {
        return this;
    }

    @Override
    @NotNull
    public SavedDataStorage getSavedDataStorage() {
        return this.savedDataStorage;
    }

    @Override
    @Nullable
    public CompoundTag getPlayerData(@Nullable String username, @NotNull UUID uuid) {
        try {
            File nameFile;
            File uuidFile = new File(this.playersDirectory, String.valueOf(uuid) + ".dat");
            if (uuidFile.exists()) {
                return NbtIo.readCompressed(Files.newInputStream(uuidFile.toPath(), new OpenOption[0]));
            }
            if (username != null && (nameFile = new File(this.playersDirectory, username + ".dat")).exists()) {
                CompoundTag tag = NbtIo.readCompressed(Files.newInputStream(nameFile.toPath(), new OpenOption[0]));
                nameFile.renameTo(uuidFile);
                return tag;
            }
        }
        catch (Exception exception) {
            LOGGER.warn("Failed to load player data for {}", (Object)username);
        }
        return null;
    }

    @FunctionalInterface
    public static interface SaveHandlerFactory {
        @NotNull
        public LevelStorageBase apply(ISaveFormat var1, File var2, String var3);
    }
}

