#version 410

out vec4 FragColor;

in vec2 TexCoords;

uniform sampler2D colortex0;
uniform sampler2D depthtex0;

// Color correction params
uniform float brightness;
uniform float contrast;
uniform float exposure;
uniform float saturation;
uniform float rMod;
uniform float gMod;
uniform float bMod;

uniform float fxaa;
uniform float heatHaze;
uniform int bloom;

uniform int dimension;
uniform int shaderEffect;

uniform float width;
uniform float height;
uniform float frameTimeCounter;

#define PI 3.14159

// FXAA defines
#define FXAA_REDUCE_MIN   (1.0/ 128.0)
#define FXAA_REDUCE_MUL   (1.0 / 8.0)
float FXAA_SPAN_MAX = 8.0 * fxaa;

vec4 applyFXAA(vec2 fragCoord, sampler2D tex, vec2 uViewportSize) {
    vec4 color;
    vec2 inverseVP = vec2(1.0 / uViewportSize.x, 1.0 / uViewportSize.y);
    vec3 rgbNW = texture(tex, (fragCoord + vec2(-1.0, -1.0)) * inverseVP).xyz;
    vec3 rgbNE = texture(tex, (fragCoord + vec2(1.0, -1.0)) * inverseVP).xyz;
    vec3 rgbSW = texture(tex, (fragCoord + vec2(-1.0, 1.0)) * inverseVP).xyz;
    vec3 rgbSE = texture(tex, (fragCoord + vec2(1.0, 1.0)) * inverseVP).xyz;
    vec3 rgbM  = texture(tex, fragCoord  * inverseVP).xyz;
    vec3 luma = vec3(0.299, 0.587, 0.114);
    float lumaNW = dot(rgbNW, luma);
    float lumaNE = dot(rgbNE, luma);
    float lumaSW = dot(rgbSW, luma);
    float lumaSE = dot(rgbSE, luma);
    float lumaM  = dot(rgbM,  luma);
    float lumaMin = min(lumaM, min(min(lumaNW, lumaNE), min(lumaSW, lumaSE)));
    float lumaMax = max(lumaM, max(max(lumaNW, lumaNE), max(lumaSW, lumaSE)));

    vec2 dir;
    dir.x = -((lumaNW + lumaNE) - (lumaSW + lumaSE));
    dir.y =  ((lumaNW + lumaSW) - (lumaNE + lumaSE));

    float dirReduce = max((lumaNW + lumaNE + lumaSW + lumaSE) *
    (0.25 * FXAA_REDUCE_MUL), FXAA_REDUCE_MIN);

    float rcpDirMin = 1.0 / (min(abs(dir.x), abs(dir.y)) + dirReduce);
    dir = min(vec2(FXAA_SPAN_MAX, FXAA_SPAN_MAX),
    max(vec2(-FXAA_SPAN_MAX, -FXAA_SPAN_MAX),
    dir * rcpDirMin)) * inverseVP;

    vec3 rgbA = 0.5 * (
    texture(tex, fragCoord * inverseVP + dir * (1.0 / 3.0 - 0.5)).xyz +
    texture(tex, fragCoord * inverseVP + dir * (2.0 / 3.0 - 0.5)).xyz);
    vec3 rgbB = rgbA * 0.5 + 0.25 * (
    texture(tex, fragCoord * inverseVP + dir * -0.5).xyz +
    texture(tex, fragCoord * inverseVP + dir * 0.5).xyz);

    float lumaB = dot(rgbB, luma);
    if ((lumaB < lumaMin) || (lumaB > lumaMax))
    color = vec4(rgbA, 1.0);
    else
    color = vec4(rgbB, 1.0);
    return color;
}

vec3 adjustBrightness(vec3 color, float value) {
    return color + value;
}

vec3 adjustContrast(vec3 color, float value) {
    return 0.5 + (1.0 + value) * (color - 0.5);
}

vec3 adjustExposure(vec3 color, float value) {
    return (1.0 + value) * color;
}

vec3 adjustSaturation(vec3 color, float value) {
    const vec3 luminosityFactor = vec3(0.2126, 0.7152, 0.0722);
    vec3 grayscale = vec3(dot(color, luminosityFactor));

    return mix(grayscale, color, 1.0 + value);
}

vec3 adjustColor(vec3 color, float rValue, float gValue, float bValue) {
    return vec3(color.r * rValue, color.g * gValue, color.b * bValue);
}

void Bloom(inout vec3 color) {
	vec3 blur = vec3(0.0);
	
	int quality = 7 * bloom;
	float range = 10 * (3 - bloom);
	
	float qh = quality / 2.0 - 0.5;
	float allStrengths = 0.0;
	for(int i=0; i < quality; i++) {
		for(int j=0; j < quality; j++) {
			vec2 offset = vec2((i - qh) * range / width, (j - qh) * range / height);
			
			float dist = 1.33 - distance(vec2(qh), vec2(i, j)) / qh;
			
			float strength = dist;
			
			blur += pow(texture(colortex0, TexCoords + offset).rgb, vec3(2.2)) * strength;
			allStrengths += strength;
		}
	}
	
	blur /= allStrengths;
	
	color.rgb *= 1.0 - (bloom * 0.1);
	color.rgb += blur.rgb * 0.15 * bloom;
}

void HeatHaze(inout vec2 coord, float amount) {
	float depth = texture(depthtex0, coord.xy).r;
	float strength = pow(depth, 500);
	strength = (strength - 0.16) * 2.0;
	strength = clamp(strength, 0.0, 1.0) * amount;
	
	vec2 wave = vec2(0.0, 0.0);
	
	float waveSize = 50;
	wave.x += sin(coord.y * 1.3 * waveSize + frameTimeCounter * 10);
	wave.x += sin(coord.y * 5.0 * waveSize + frameTimeCounter * 14);
	wave.y += sin(coord.x * 1.0 * waveSize + frameTimeCounter * 5) * 2;
	
	coord += wave * strength * 0.0001;
}


vec3 colorEffects(vec3 tex) {
    if (shaderEffect == 1) {
        return vec3(tex.r, 0, 0);
    }
    return tex;
}


void main() {
    vec3 color = vec3(0.0);
	vec2 coord = TexCoords;

	if(heatHaze > 0) {
		HeatHaze(coord, heatHaze);
	}

    if (fxaa > 0.01) {
        color = applyFXAA(vec2(coord.x * width, coord.y * height), colortex0, vec2(width, height)).rgb;
    } else {
        color = texture(colortex0, coord.xy).rgb;
    }

	if(bloom > 0) {
		Bloom(color);
	}

    // Perform color correction
    color = adjustBrightness(color, brightness);
    color = adjustContrast(color, contrast);
    color = adjustExposure(color, exposure);
    color = adjustSaturation(color, saturation);
    color = adjustColor(color, rMod, gMod, bMod);
    color = colorEffects(color);

    FragColor = vec4(color, 1.0);

//    float depth = texture(depthtex0, TexCoords).r;
//    depth = pow(depth, 1000);
//    FragColor = vec4(depth, depth, depth, 1.0);
}