/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.core.world.chunk;

import com.mojang.logging.LogUtils;
import com.mojang.nbt.NbtIo;
import com.mojang.nbt.tags.CompoundTag;
import com.mojang.nbt.tags.ListTag;
import com.mojang.nbt.tags.Tag;
import it.unimi.dsi.fastutil.ints.Int2ObjectArrayMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.Arrays;
import java.util.List;
import net.minecraft.core.NextTickListEntry;
import net.minecraft.core.block.entity.TileEntity;
import net.minecraft.core.block.entity.TileEntityDispatcher;
import net.minecraft.core.entity.Entity;
import net.minecraft.core.entity.EntityDispatcher;
import net.minecraft.core.util.helper.LightIndexHelper;
import net.minecraft.core.world.World;
import net.minecraft.core.world.chunk.Chunk;
import net.minecraft.core.world.chunk.ChunkLoader;
import net.minecraft.core.world.chunk.ChunkSection;
import net.minecraft.core.world.chunk.reader.ChunkReader;
import net.minecraft.core.world.chunk.reader.ChunkReaderLegacy;
import net.minecraft.core.world.chunk.reader.ChunkReaderVersion1;
import net.minecraft.core.world.chunk.reader.ChunkReaderVersion2;
import net.minecraft.core.world.chunk.reader.ChunkReaderVersion3;
import net.minecraft.core.world.chunk.writer.ChunkWriter;
import net.minecraft.core.world.data.ChunkNibbleArray;
import net.minecraft.core.world.data.ChunkUnsignedByteArray;
import net.minecraft.core.world.save.LevelData;
import org.slf4j.Logger;

public class ChunkLoaderLegacy
implements ChunkLoader {
    private static final Logger LOGGER = LogUtils.getLogger();
    private File worldDir;
    private boolean createIfNecessary;

    public ChunkLoaderLegacy(File worldDir, boolean createIfNecessary) {
        this.worldDir = worldDir;
        this.createIfNecessary = createIfNecessary;
    }

    private File chunkFileForXZ(int x, int z) {
        String fileName = "c." + Integer.toString(x, 36) + "." + Integer.toString(z, 36) + ".dat";
        String xDirName = Integer.toString(x & 0x3F, 36);
        String zDirName = Integer.toString(z & 0x3F, 36);
        File dir = new File(this.worldDir, xDirName);
        if (!dir.exists()) {
            if (this.createIfNecessary) {
                dir.mkdir();
            } else {
                return null;
            }
        }
        if (!(dir = new File(dir, zDirName)).exists()) {
            if (this.createIfNecessary) {
                dir.mkdir();
            } else {
                return null;
            }
        }
        if (!(dir = new File(dir, fileName)).exists() && !this.createIfNecessary) {
            return null;
        }
        return dir;
    }

    @Override
    public Chunk loadChunk(World world, int x, int z) throws IOException {
        File file = this.chunkFileForXZ(x, z);
        if (file != null && file.exists()) {
            try {
                InputStream fileStream = Files.newInputStream(file.toPath(), new OpenOption[0]);
                CompoundTag tag = NbtIo.readCompressed(fileStream);
                if (!tag.containsKey("Level")) {
                    LOGGER.warn("Chunk file at {},{} is missing level data, skipping", (Object)x, (Object)z);
                    return null;
                }
                if (!tag.getCompound("Level").containsKey("Blocks")) {
                    LOGGER.warn("Chunk file at {},{} is missing block data, skipping", (Object)x, (Object)z);
                    return null;
                }
                Chunk chunk = ChunkLoaderLegacy.loadChunkIntoWorldFromCompound(world, tag.getCompound("Level"));
                if (!chunk.isAtLocation(x, z)) {
                    LOGGER.warn("Chunk file at {},{} is in the wrong location; relocating. (Expected {}, {}, got {}, {})", x, z, x, z, chunk.pos.x, chunk.pos.z);
                    tag.putInt("xPos", x);
                    tag.putInt("zPos", z);
                    chunk = ChunkLoaderLegacy.loadChunkIntoWorldFromCompound(world, tag.getCompound("Level"));
                }
                chunk.fixMissingBlocks();
                return chunk;
            }
            catch (Exception exception) {
                LOGGER.error("Exception while attempting to load chunk at X:{}, Z:{}", x, z, exception);
            }
        }
        return null;
    }

    @Override
    public void saveChunk(World world, Chunk chunk) throws IOException {
        world.checkSessionLock();
        File chunkFile = this.chunkFileForXZ(chunk.pos.x, chunk.pos.z);
        if (chunkFile.exists()) {
            LevelData levelData = world.getLevelData();
            levelData.setSizeOnDisk(levelData.getSizeOnDisk() - chunkFile.length());
        }
        try {
            File tmpChunkFile = new File(this.worldDir, "tmp_chunk.dat");
            OutputStream fileStream = Files.newOutputStream(tmpChunkFile.toPath(), new OpenOption[0]);
            CompoundTag levelTag = new CompoundTag();
            CompoundTag chunkDataTag = new CompoundTag();
            levelTag.put("Level", chunkDataTag);
            ChunkLoaderLegacy.storeChunkInCompound(chunk, world, chunkDataTag);
            NbtIo.writeCompressed(levelTag, fileStream);
            fileStream.close();
            if (chunkFile.exists()) {
                chunkFile.delete();
            }
            tmpChunkFile.renameTo(chunkFile);
            LevelData levelData = world.getLevelData();
            levelData.setSizeOnDisk(levelData.getSizeOnDisk() + chunkFile.length());
        }
        catch (Exception exception) {
            LOGGER.error("Exception while attempting to save chunk at X:{}, Z:{}", chunk.pos.x, chunk.pos.z, exception);
        }
    }

    @Override
    public boolean isSaving() {
        return false;
    }

    public static void storeChunkInCompound(Chunk chunk, World world, CompoundTag tag) {
        world.checkSessionLock();
        ChunkWriter writer = new ChunkWriter(world, tag);
        tag.putInt("Version", 3);
        writer.putXPos(chunk.pos.x);
        writer.putZPos(chunk.pos.z);
        writer.putTicksOnUnload(world.getWorldTime());
        writer.putHeightMap(chunk.heightMap);
        writer.putAverageBlockHeight(chunk.averageBlockHeight);
        writer.putTerrainPopulated(chunk.isTerrainPopulated);
        writer.putTemperatureMap(chunk.temperature);
        writer.putHumidityMap(chunk.humidity);
        writer.putTickList(chunk.tickList);
        for (int i = 0; i < 16; ++i) {
            ChunkLoaderLegacy.storeChunkSectionInCompound(chunk.getSection(i), writer);
        }
        chunk.hasEntities = false;
        ListTag entityListTag = new ListTag();
        for (int i = 0; i < 16; ++i) {
            ChunkSection section = chunk.getSection(i);
            List<Entity> entities = section.entities;
            for (int j = 0; j < entities.size(); ++j) {
                Entity entity = entities.get(j);
                chunk.hasEntities = true;
                CompoundTag entityTag = new CompoundTag();
                if (!entity.save(entityTag)) continue;
                entityListTag.addTag(entityTag);
            }
        }
        tag.put("Entities", entityListTag);
        ListTag tileEntityListTag = new ListTag();
        for (TileEntity tileEntity : chunk.tileEntityMap.values()) {
            CompoundTag tileEntityTag = new CompoundTag();
            tileEntity.writeToNBT(tileEntityTag);
            tileEntityListTag.addTag(tileEntityTag);
        }
        tag.put("TileEntities", tileEntityListTag);
        writer.putBiomeRegistry();
    }

    private static void storeChunkSectionInCompound(ChunkSection section, ChunkWriter writer) {
        if (section.blocks != null) {
            writer.putBlocks(section.yPosition, section.blocks);
        }
        if (section.data != null) {
            writer.putData(section.yPosition, section.data);
        }
        if (section.lightData != null) {
            byte[] blockLightData = new byte[section.lightData.length / 2];
            byte[] skyLightData = new byte[section.lightData.length / 2];
            for (int i = 0; i < section.lightData.length / 2; ++i) {
                byte lightIndex0 = section.lightData[i * 2];
                byte lightIndex1 = section.lightData[i * 2 + 1];
                int block0 = LightIndexHelper.blockLightFromIndex(lightIndex0);
                int block1 = LightIndexHelper.blockLightFromIndex(lightIndex1);
                int sky0 = LightIndexHelper.skyLightFromIndex(lightIndex0);
                int sky1 = LightIndexHelper.skyLightFromIndex(lightIndex1);
                blockLightData[i] = (byte)(block1 << 4 | block0);
                skyLightData[i] = (byte)(sky1 << 4 | sky0);
            }
            writer.putSkyLight(section.yPosition, skyLightData);
            writer.putBlockLight(section.yPosition, blockLightData);
        }
        writer.putBiomeMap(section.yPosition, section.biome);
    }

    public static Chunk loadChunkIntoWorldFromCompound(World world, CompoundTag tag) {
        ListTag tileEntityListTag;
        ListTag entityListTag;
        int version = tag.getIntegerOrDefault("Version", -1);
        ChunkReader reader = ChunkLoaderLegacy.getChunkReaderByVersion(world, tag, version);
        int x = reader.getX();
        int z = reader.getZ();
        Chunk chunk = new Chunk(world, x, z);
        chunk.ticksOnUnload = reader.getTicksOnUnload();
        chunk.heightMap = reader.getHeightMap();
        chunk.averageBlockHeight = reader.getAverageBlockHeight();
        chunk.isTerrainPopulated = reader.getIsTerrainPopulated();
        chunk.temperature = reader.getTemperatureMap();
        chunk.humidity = reader.getHumidityMap();
        chunk.tickList = reader.getTickList();
        Int2ObjectMap<String> biomeRegistry = reader.getBiomeRegistry();
        for (int i = 0; i < 16; ++i) {
            ChunkLoaderLegacy.loadChunkSectionFromCompound(chunk.getSection(i), reader, biomeRegistry);
        }
        if (chunk.heightMap == null) {
            chunk.heightMap = new short[256];
            chunk.recalcHeightmapAndLightmap();
        }
        if (chunk.temperature == null || chunk.temperature.length == 0) {
            chunk.temperature = new double[256];
            Arrays.fill(chunk.temperature, Double.NEGATIVE_INFINITY);
        }
        if (chunk.humidity == null || chunk.humidity.length == 0) {
            chunk.humidity = new double[256];
            Arrays.fill(chunk.humidity, Double.NEGATIVE_INFINITY);
        }
        if (chunk.tickList == null) {
            chunk.tickList = new Int2ObjectArrayMap<NextTickListEntry>();
        }
        if ((entityListTag = tag.getList("Entities")) != null) {
            for (Tag<?> entityTagBase : entityListTag) {
                if (!(entityTagBase instanceof CompoundTag)) continue;
                CompoundTag entityTag = (CompoundTag)entityTagBase;
                Entity entity = EntityDispatcher.getInstance().createEntityFromNBT(entityTag, world);
                chunk.hasEntities = true;
                if (entity == null) continue;
                chunk.addEntity(entity);
            }
        }
        if ((tileEntityListTag = tag.getList("TileEntities")) != null) {
            for (Tag<?> tileEntityTagBase : tileEntityListTag) {
                CompoundTag tileEntityTag;
                TileEntity tileEntity;
                if (!(tileEntityTagBase instanceof CompoundTag) || (tileEntity = TileEntityDispatcher.createAndLoadEntity(tileEntityTag = (CompoundTag)tileEntityTagBase)) == null) continue;
                chunk.addTileEntity(tileEntity);
            }
        }
        return chunk;
    }

    public static void loadChunkSectionFromCompound(ChunkSection section, ChunkReader reader, Int2ObjectMap<String> biomeRegistry) {
        int i;
        section.blocks = reader.getBlocks(section.yPosition);
        section.data = reader.getData(section.yPosition);
        ChunkNibbleArray skylightMap = reader.getSkyLight(section.yPosition);
        ChunkNibbleArray blocklightMap = reader.getBlockLight(section.yPosition);
        section.lightData = new byte[4096];
        if (skylightMap != null && skylightMap.data != null) {
            for (i = 0; i < section.lightData.length / 2; ++i) {
                int n = i * 2;
                section.lightData[n] = (byte)(section.lightData[n] | (byte)((skylightMap.data[i] & 0xF) << 4));
                int n2 = i * 2 + 1;
                section.lightData[n2] = (byte)(section.lightData[n2] | (byte)(((skylightMap.data[i] & 0xF0) >> 4 & 0xF) << 4));
            }
        }
        if (blocklightMap != null && blocklightMap.data != null) {
            for (i = 0; i < section.lightData.length / 2; ++i) {
                int n = i * 2;
                section.lightData[n] = (byte)(section.lightData[n] | (byte)(blocklightMap.data[i] & 0xF));
                int n3 = i * 2 + 1;
                section.lightData[n3] = (byte)(section.lightData[n3] | (byte)((blocklightMap.data[i] & 0xF0) >> 4 & 0xF));
            }
        }
        section.biome = reader.getBiomeMap(section.yPosition, biomeRegistry);
        if (section.data != null && !section.data.isValid()) {
            section.data = new ChunkUnsignedByteArray(16, 16, 16);
        }
        if (section.biome == null || section.biome.length != 512) {
            section.biome = new byte[512];
            Arrays.fill(section.biome, (byte)-1);
        }
    }

    private static ChunkReader getChunkReaderByVersion(World world, CompoundTag tag, int version) {
        switch (version) {
            case 1: {
                return new ChunkReaderVersion1(world, tag);
            }
            case 2: {
                return new ChunkReaderVersion2(world, tag);
            }
            case 3: {
                return new ChunkReaderVersion3(world, tag);
            }
        }
        return new ChunkReaderLegacy(world, tag);
    }
}

